C      *****************************************************************
C      * Subroutine WthrKT                                             *
C      * Calculates the amount of each mineral which is weathered away *
C      * in each time step and adjusts the known totals for components *
C      * contained in those minerals.                                  *
C      *                                                               *
C      * Variables:                                                    *
C      * BlkDens   -Input - A vector, of size NLayers, of the bulk     *
C      *                    density of the soil for each layer.  Input *
C      *                    from TREGRO, or read into the model by the *
C      *                    routine InParam.  Units are g/cm**3.       *
C      *                    (Common block VModel, file VModel.f)       *
C      * C         -Local - Component index.  Unitless.                *
C      * CatchArea -Input - A vector, of size NLayers, of the catchment*
C      *                    area for each layer.  Input from TREGRO, or*
C      *                    read into the model by the routine InParam.*
C      *                    Units are m**2.                            *
C      *                    (Common block VModel, file VModel.f)       *
C      * Cm3_m3    -Local - Conversion constant from 1/cm**3 to 1/m**3.*
C      *                    Units are (cm**3/m**3).
C      * ElapT     -Input - The elapsed time since the last call, i.e.,*
C      *                    the model timestep.  (This is model time,  *
C      *                    not real time.)  Input from TREGRO, or read*
C      *                    into the model by the routine InParam.     *
C      *                    Units are s.                               *
C      *                    (Common block VModel, file VModel.f)       *
C      * GoAhead   -Output- Indicates whether any errors were found.   *
C      *                    (GoAhead=1, no errors found; GoAhead=0,    *
C      *                    errors found.)  Unitless.                  *
C      * HIndex    -Local - The hydrogen ion index.  Unitless.         *
C      * HOrder    -Input - A vector, of size NMnrls, of reaction      *
C      *                    orders with respect to hydrogen ion        *
C      *                    concentration, for each mineral.  Unitless.*
C      *                    (Common block VModel, file VModel.f)       *
C      * L         -Local - Layer index.  Unitless.                    *
C      * M         -Local - Mineral index.  Unitless.                  *
C      * MinFrac   -Input - A matrix, of size NMnrls by NLayers, of    *
C      *                    the amount of each mineral that is in each *
C      *                    layer as a fraction of soil weight. Units  *
C      *                    are g/g and the values range between zero  *
C      *                    and one.                                   *
C      *                    (Common block VModel, file VModel.f)       *
C      * MinMW     -Input - A vector, of size NMnrls, of formula       *
C      *                    weights for each mineral.  Units are       *
C      *                    g/mol.                                     *
C      *                    (Common block VModel, file VModel.f)       *
C      * MinSC     -Input - An array, of size NMnrls by NComp, of the  *
C      *                    stoichiometric coefficient of each         *
C      *                    component for each mineral.  Unitless.     *
C      *                    (Common block VModel, file VModel.f)       *
C      * MOrder    -Input - A vector, of size NMnrls, of reaction      *
C      *                    orders with respect to MinFrac, for each   *
C      *                    mineral.  Unitless.                        *
C      *                    (Common block VModel, file VModel.f)       *
C      * Name      -Local - Used for passing data to WhereIs function. *
C      *                    Unitless.                                  *
C      * NComp     -Input - The number of components in the system.    *
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * NLayers   -Input - The number of soil layers in the system.   *
C      *                    Input from TREGRO, or read into the model  *
C      *                    by the routine InParam.  Unitless.         *
C      *                    (Common block VPos, file VModel.f)         *
C      * NMnrls    -Input - The number of minerals in the system.      *
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * SpConc    -Input - A matrix, of size NSpecies by NLayers, of  *
C      *                    species concentrations, one set for each   *
C      *                    layer. Units are mol/L.                    *
C      *                    (Common block VModel, file VModel.f)       *
C      * Thickness -Input - A vector, of size NLayers, of thicknesses  *
C      *                    of each soil layer.  Input from TREGRO, or *
C      *                    read into the model by the routine InParam.*
C      *                    Units are m.                               *
C      *                    (Common block VModel, file VModel.f)       *
C      * WAmt      -Local - The amount of the current mineral in the   *
C      *                    current layer weathered during this time   *
C      *                    step.  Units are mol.                      *
C      * Weight    -Local - A vector, of size NLayers, of the weight of*
C      *                    each soil layer. Units are g.              *
C      * WComp     -Output- A vector, of size NComp, of the amount of  *
C      *                    component weathered during this time step. *
C      *                    Units are mol.                             *
C      *                    (Common block VModel, file VModel.f)       *
C      * WFrac     -Local - The mass of mineral weathered during this  *
C      *                    timestep as a fraction of the weight of    *
C      *                    soil in the current layer.  Unitless.      *
C      * WhereIs   -Fcn   - Returns the index of requested species. If *
C      *                    the species is not found a -1 is returned. *
C      *                    Unitless.                                  *
C      * WMin      -Output- A vector, of size NMrls, of the amount of  *
C      *                    each mineral weathered during this         *
C      *                    time step.  Units are mol.                 *
C      *                    (Common block VModel, file VModel.f)       *
C      * WRate     -Input - A vector, of size NComp, of rate constants *
C      *                    for each mineral.  Value and units must be *
C      *                    consistent with values of HOrder and       *
C      *                    MOrder. For example: WRate is the fraction *
C      *                    of mineral weathered per time step when    *
C      *                    MOrder=1 and HOrder=0.                     *
C      *                    (Common block VModel, file VModel.f)       *
C      * WthrC     -Local - The amount of the current component in the *
C      *                    in the current layer released due to the   *
C      *                    weathering of the current mineral.  Units  *
C      *                    are mol.                                   *
C      * YASOut    -Input - The output file number for the YASEQL      *
C      *                    model.  Unitless.                          *
C      *                    (Common block VModel, file VModel.f)       *
C      *****************************************************************
       SUBROUTINE WTHRKT(GOAHEAD)
							IMPLICIT NONE
							INCLUDE 'VMODEL.INC'
							INTEGER C, GOAHEAD, HINDEX, L, M, WHEREIS
							REAL*8 CM3_M3/1E6/, WAMT, WEIGHT(15), WFRAC, WTHRC
							CHARACTER*8 NAME

C      *-------------------------------------------------------*
C      * Initialize variable to indicate no errors encountered *
C      *-------------------------------------------------------*
       GOAHEAD = 1
	   
C      *------------------------------------------------------------------*
C      * Find the hydrogen ion index into the species concentration array *	   
C      *------------------------------------------------------------------*
       NAME = 'H'
       HINDEX = WHEREIS(NAME)
	      IF (HINDEX.EQ.-1) THEN
	         GOAHEAD = 0
	         WRITE(6,*) '--->Error in WthrKT routine, file: WthrKT'
										WRITE(6,*) 'Hydrogen ion ("H") index not found'
	         WRITE(YASOUT,*) 
     >    '--->Error in WthrKT routine, file: WthrKT'
										WRITE(YASOUT,*)
     >    'Hydrogen ion ("H") index not found'
										GOTO 9999
	      ENDIF

C      *----------------------------------------------*
C      * Calculate the weight of each layer, in grams *
C      *----------------------------------------------*
       DO 50 L=1, NLAYERS
          WEIGHT(L) = BLKDENS(L) * CATCHAREA(L) * THICKNESS(L) * CM3_M3
   50  CONTINUE

C      *---------------------------------------------------------*
C      * Clear the total number of moles of each mineral lost to *
C      * weathering from all layers during this time step.       *
C      *---------------------------------------------------------*
       DO 70 M=1, NMNRLS
	         WMIN(M) = 0
   70  CONTINUE		  

C      *-------------------------------------------------------------*
C      * Clear the total number of moles of each component weathered *
C      * from all minerals during this time step.                    *
C      *-------------------------------------------------------------*
       DO 80 C=1, NCOMP
	         WCOMP(C) = 0
   80  CONTINUE		  

C      *--------------------------*
C      * Loop over all minerals   *
C      *--------------------------*
       DO 100 M=1, NMNRLS	 

C         *----------------------*
C         * Loop over all layers *
C         *----------------------*
          DO 200 L=1, NLAYERS
	   
C            *--------------------------------------------------------*
C            * Calculate the mass of the mineral that is weathered as *
C            * a fraction of the mass of soil in the current layer.   *
C            * If both the mineral fraction and its power (MOrder)    *
C            * are zero, set the quantity MinFrac ** MOrder to 1. The *
C            * limit as MinFrac approaches zero is 1.                 *
C            *--------------------------------------------------------*
             IF ((MINFRAC(M,L).EQ.0).AND.(MORDER(M).EQ.0)) THEN
			             WFRAC = WRATE(M) * SPCONC(HINDEX,L) ** HORDER(M) * 
     >                  ELAPT
		           ELSE
                WFRAC = WRATE(M) * MINFRAC(M,L) ** MORDER(M) * 
     >                  SPCONC(HINDEX,L) ** HORDER(M) * ELAPT	   
			          ENDIF
			 
C            *----------------------------------------------------*
C            * Adjust the amount of mineral in this soil layer to *
C            * account for weathering loss. Do not allow the      *
C            * mineral fraction to become negative.               *
C            *----------------------------------------------------* 
             IF (WFRAC.GT.MINFRAC(M,L)) THEN
			             WRITE(6,*) 'In Weathering routine, Wfrac adjusted'
			            	WRITE(6,*) 'so that minfrac >= 0'
			             WRITE(YASOUT,*)
     >          'In Weathering routine, Wfrac adjusted'
			            	WRITE(YASOUT,*) 'so that minfrac >= 0'
		        	    WFRAC = MINFRAC(M,L)
		           ENDIF
             MINFRAC(M,L) = MINFRAC(M,L) - WFRAC

C            *-----------------------------------------------*         
C            * Calculate the number of moles of this mineral *
C            * weathered in this layer.                      *
C            *-----------------------------------------------*         
             WAMT = WFRAC * WEIGHT(L) / MINMW(M)

C            *------------------------------------------------------*         
C            * Sum up the number of moles of this mineral weathered * 
C            * in all layers in this time step. (For output only)   *
C            *------------------------------------------------------* 
             WMIN(M) = WMIN(M) + WAMT

C            *---------------------------------------------------------* 
C            * Calculate the moles of each component that was weathered*
C            * from this mineral and adjust each component's known     *
C            * total accordingly.  Save the total amount of each       *
C            * component weathered during each time step.              *
C            *---------------------------------------------------------* 
             DO 300 C=1, NCOMP
			             WTHRC = WAMT * MINSC(M,C)
																KT(C,L) = KT(C,L) + WTHRC
																WCOMP(C) = WCOMP(C) + WTHRC
  300        CONTINUE			
  200     CONTINUE
  100  CONTINUE
C      *--------------*
C      * Escape hatch *
C      *--------------*
 9999  CONTINUE

       RETURN
	      END
C      *****************************************************************
C      *                    END SUBROUTINE                             *
C      *****************************************************************
